# marc-get.pl - print to STDOUT MARC records in communications format based on ISBN numbers

# exampe usage: marc-get.pl 1558322612 > data.mrc

# Eric Lease Morgan (emorgan@nd.edu)
# September 1, 2003

# based on GET-MARC-ISBN by Chris Biemesderfer, see:
#
#  http://marcpm.sourceforge.net/MARC/Doc/Tutorial.html

# use the necessaray modules
use strict;
use Carp;
use Net::Z3950;
use MARC::Record;

# define some contants
my $DATABASE = 'voyager';
my $SERVER   = 'z3950.loc.gov';
my $PORT     = '7090';

# check for command line input, a string of ISBN numbers
exit if ($#ARGV < 0);

# We handle multiple ISBNs in the same query by assembling a
# (potentially very large) search string with Prefix Query Notation
# that ORs the ISBN-bearing attributes.
#
# For purposes of automation, we want to request batches of many MARC
# records.  I am not a Z39.50 weenie, though, and I don't know
# offhand if there is a limit on how big a PQN query can be...

# initialize a z39.50 query
my $zq = "\@attr 1=7 " . pop();

# populate the query with ISBN numbers
while (@ARGV) { $zq = '@or @attr 1=7 '. pop() ." $zq" }

# create a z39.50 object
my $mgr = Net::Z3950::Manager->new(databaseName => $DATABASE);

# assign the object some z39.50 characteristics
$mgr->option(elementSetName => "f");
$mgr->option(preferredRecordSyntax => Net::Z3950::RecordSyntax::USMARC);
	
# connect to the server and check for success
my $conn = $mgr->connect($SERVER, $PORT);
croak "Unable to connect to server $SERVER:$PORT" if ! defined($conn);
	
# search
my $rs = $conn->search($zq);

# get and report on the number of records found
my $numrec = $rs->size();
print STDERR "$numrec record(s) found\n";
	
# process each found record
for (my $i = 1; $i <= $numrec; $i++) {

	# initilize an individual record; get it; print it
	my $zrec = $rs->record($i);
	my $mrec = MARC::Record->new_from_usmarc($zrec->rawdata());
	print $mrec->as_usmarc();

} 

# done
exit;
